"use client";
// Start import
import { useState, useEffect } from "react";
import { Button } from "../../../../components/ui/button";
import { Label } from "../../../../components/ui/label";
import Select from "react-select";
import instance from "../../../../lib/axiosInstance";
import { useForm, Controller } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as yup from "yup";
import { Loader2 } from "lucide-react";
import { useCrudOperation } from "../../../context/CrudOperationContect";
import { toast } from "sonner";
import DatePicker from "react-datepicker";
import "react-datepicker/dist/react-datepicker.css";
import { registerLocale } from "react-datepicker";
import fr from "date-fns/locale/fr";
import { DialogClose } from "../../../../components/ui/dialog";
import { Input } from "../../../../components/ui/input";
registerLocale("fr", fr);
import { selectStyles } from "../../../styles/selectStyles";
// End import

// Schéma de validation YUP
const schema = yup.object().shape({
  amount: yup
    .number()
    .transform((value) => (isNaN(value) ? undefined : value))
    .required("Le montant est requis")
    .min(0, "Le montant doit être supérieur à 0"),
  terms: yup
    .number()
    .transform((value) => (isNaN(value) ? undefined : value))
    .nullable()
    .min(1, "Le nombre de modalités doit être supérieur à 0"),
  due_date: yup.date().required("La date limite est requise"),
  level: yup.object().required("Le niveau est requis"),
  fee_type: yup.object().required("Le type de frais est requis"),
});

// Export
export const Form = ({ closeModal, rowData }) => {
  // Utilisation du contexte pour accéder aux données et fonctions
  const { addOperation, updateOperation } = useCrudOperation();
  const [levels, setLevels] = useState([]);
  const [feetTypes, setFeetTypes] = useState([]);

  //
  const fetchLevels = async () => {
    try {
      const response = await instance.get("settings/levels");
      const rawData = response.data.response.data;
      const formattedOptions = rawData.map((level) => ({
        value: level.id.toString(),
        label: level.name,
      }));
      setLevels(formattedOptions);
    } catch (error) {
      console.error("Erreur chargement niveaux :", error);
    }
  };

  //
  const fetchFeetTypes = async () => {
    try {
      const response = await instance.get("fee-types");
      const rawData = response.data.response.data;
      const formattedOptions = rawData.map((row) => ({
        value: row.id.toString(),
        label: row.name,
      }));
      setFeetTypes(formattedOptions);
    } catch (error) {
      console.error("Erreur chargement niveaux :", error);
      toast.error("Erreur", {
        description: "Impossible de charger les niveaux",
        position: "top-center",
      });
    }
  };

  // Format pre-selected data
  const defaultSelectedLevel = rowData?.level ? {
    value: rowData.level.id.toString(),
    label: rowData.level.name,
  } : null;
  const defaultSelectedFeeType = rowData?.fee_type ? {
    value: rowData.fee_type.id.toString(),
    label: rowData.fee_type.name,
  } : null;

  // Set useEffect hook
  useEffect(() => {
    fetchLevels();
    fetchFeetTypes();
  }, []);

  // Define react-hook-form
  const {
    control,
    register,
    handleSubmit,
    formState: { errors, isSubmitting },
  } = useForm({
    resolver: yupResolver(schema),
    defaultValues: {
      amount: rowData?.amount || '',
      terms: rowData?.terms || '',
      due_date: rowData?.due_date ? new Date(rowData.due_date) : null,
      level: defaultSelectedLevel,
      fee_type: defaultSelectedFeeType,
    },
  });

  // Set method to add classrooms
  const onSubmit = async (data) => {
    try {
      console.log(data);
      // Formatage des données pour l'API
      const formattedData = {
        amount: parseFloat(data.amount),
        terms: data.terms ? parseInt(data.terms) : null,
        due_date: data.due_date
          ? data.due_date.toISOString().split("T")[0]
          : null,
        level_id: data.level.value,
        fee_type_id: data.fee_type.value,
      };

      // Set query
      // const query = await instance.post("fees", formattedData);

      // Add data to state
      addOperation(query.data.response);

      // Set success message
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });

      // Ferme le modal après soumission réussie
      closeModal();
    } catch (error) {
      console.log(error);
      // Gestion des erreurs de validation
      const errors = error.response?.data?.response;
      if (errors) {
        Object.entries(errors).map(([field, messages]) => {
          toast.error("Erreur", {
            description: messages,
            position: "top-center",
            style: {
              backgroundColor: "#E11409FF",
              color: "white",
            },
          });
        });
      }
    }
  };

  // Set method to add classrooms
  const onUpdate = async (data) => {
    try {
      // Formatage des données pour l'API
      const formattedData = {
        amount: parseFloat(data.amount),
        terms: data.terms ? parseInt(data.terms) : null,
        due_date: data.due_date
          ? data.due_date.toISOString().split("T")[0]
          : null,
        level_id: data.level.value,
        fee_type_id: data.fee_type.value,
      };

      // Set query
      const query = await instance.put("fees/" + rowData?.id, formattedData);
      updateOperation(query.data.response);
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });

      // Ferme le modal après soumission réussie
      closeModal();
    } catch (error) {
      // Gestion des erreurs de validation
      const errors = error.data?.response;
      if (errors) {
        Object.entries(errors).map(([field, messages]) => {
          toast.error("Erreur", {
            description: messages,
            position: "top-center",
            style: {
              backgroundColor: "#E11409FF",
              color: "white",
            },
          });
        });
      }
    }
  };

  // Return
  return (
    <form
      onSubmit={rowData?.id ? handleSubmit(onUpdate) : handleSubmit(onSubmit)}
    >
      <div className="">
        <Label htmlFor="amount" className="text-right mb-2">
          Montant <span className="text-red-500">*</span>
        </Label>
        <div className="relative">
          <Input
            id="amount"
            type="number"
            step="0.01"
            min="0"
            {...register("amount", {
              setValueAs: (value) => (value === "" ? undefined : parseFloat(value)),
            })}
            placeholder="00.00"
            className="text-end"
          />
        </div>
        <span className="text-red-500 text-sm">{errors.amount?.message}</span>
      </div>

      <div className="py-4 grid grid-cols-4 gap-4">
        <div className="col-span-2">
          <Label htmlFor="terms" className="text-right mb-2">
            Modalité
          </Label>
          <Input
            id="terms"
            type="number"
            min="1"
            {...register("terms", {
              setValueAs: (value) => (value === "" ? null : parseInt(value)),
            })}
            placeholder="Modalité de paiement"
          />
          <span className="text-red-500 text-sm">{errors.terms?.message}</span>
        </div>
        <div className="col-span-2">
          <Label htmlFor="due_date" className="text-right mb-2">
            Date limite de paiement <span className="text-red-500">*</span>
          </Label>
          <Controller
            name="due_date"
            control={control}
            render={({ field }) => (
              <DatePicker
                className="px-4 py-3 bg-white w-full text-sm border-2 border-gray-200 focus:border-[#fb923d] rounded-md outline-none text-black"
                selected={field.value}
                onChange={(date) => field.onChange(date)}
                placeholderText="Choisir une date"
                dateFormat="dd/MM/yyyy"
                locale="fr"
                minDate={new Date()}
              />
            )}
          />
          <span className="text-red-500 text-sm">{errors.due_date?.message}</span>
        </div>
      </div>

      <div className="flex gap-5">
        <div className="w-full">
          <Label htmlFor="level" className="text-right mb-2">
            Niveau <span className="text-red-500">*</span>
          </Label>
          <Controller
            name="level"
            control={control}
            render={({ field }) => (
              <Select
                options={levels}
                value={field.value}
                onChange={(options) => field.onChange(options)}
                placeholder="Choisir un niveau"
                classNamePrefix="select"
                styles={selectStyles}
              />
            )}
          />
          <span className="text-red-500 text-sm">{errors.level?.message}</span>
        </div>

        <div className="w-full">
          <Label htmlFor="fee_type" className="text-right mb-2">
            Type de frais <span className="text-red-500">*</span>
          </Label>
          <Controller
            name="fee_type"
            control={control}
            render={({ field }) => (
              <Select
                options={feetTypes}
                value={field.value}
                
                onChange={(options) => field.onChange(options)}
                placeholder="Choisir type de frais"
                classNamePrefix="select"
                styles={selectStyles}
              />
            )}
          />
          <span className="text-red-500 text-sm">{errors.fee_type?.message}</span>
        </div>
      </div>

      <div className="flex justify-center items-center mt-4 gap-2">
        <DialogClose asChild>
          <Button variant="destructive" className="h-12">
            Annuler
          </Button>
        </DialogClose>
        <Button type="submit" disabled={isSubmitting} className="h-12">
          {isSubmitting ? (
            <Loader2 className="animate-spin w-5 h-5 mr-2" />
          ) : (
            "Enregistrer"
          )}
        </Button>
      </div>
    </form>
  );
};
