"use client";
// Start import
import { useState, useEffect } from "react";
import { Button } from "../../../../components/ui/button";
import { Label } from "../../../../components/ui/label";
import Select from "react-select";
import instance from "../../../../lib/axiosInstance";
import { useForm, Controller } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as yup from "yup";
import { Loader2 } from "lucide-react";
import { useCrudOperation } from "../../../context/CrudOperationContect";
import { toast } from "sonner";
// End import

// Schéma de validation YUP
const schema = yup.object().shape({
  first_name: yup.string().required(""),
  last_name: yup.string().required(""),
  email: yup.string().required(""),
  phone: yup.string().required(""),
  subjects: yup.array().required(),
});

// Export
export const Form = ({ closeModal, rowData }) => {
  // Set state for levelsOption
  const [levelTeacher, setTeacher] = useState([]);

  // Utilisation du contexte pour accéder aux données et fonctions
  const { addOperation, updateOperation } = useCrudOperation();

  // Format pre-selected data
  const defaultSelectedLevels = rowData?.subjects.map((subject) => ({
    value: subject.id,
    label: subject.name,
  }));

  // Set useEffect hook
  useEffect(() => {
    const fetchLevelOptions = async () => {
      try {
        const response = await instance.get("settings/subjects");
        const rawData = response.data.response.data;
        const formattedOptions = rawData.map((level) => ({
          value: level.id.toString(),
          label: level.name,
        }));
        setTeacher(formattedOptions);
      } catch (error) {
        console.error("Erreur chargement niveaux :", error);
        toast.error("Erreur", {
          description: "Impossible de charger les niveaux",
          position: "top-center",
        });
      }
    };
    fetchLevelOptions();
  }, []);

  // Define react-hook-form
  const {
    control,
    register,
    handleSubmit,
    formState: { errors, isSubmitting },
  } = useForm({
    resolver: yupResolver(schema),
    defaultValues: {
      first_name: rowData?.first_name,
      last_name: rowData?.last_name,
      email: rowData?.email,
      phone: rowData?.phone,
      subjects: defaultSelectedLevels,
    },
  });

  // Set method to add
  const onSubmit = async (data) => {
    try {
      // Formatage des données pour l'API
      const formattedData = {
        first_name: data.first_name,
        last_name: data.last_name,
        email: data.email,
        phone: data.phone,
        subjects: data.subjects.map((subject) => subject.value),
      };
      const query = await instance.post("teachers", formattedData); // Correction de l'endpoint
      addOperation(query.data.response);
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });

      // Ferme le modal après soumission réussie
      closeModal();
    } catch (error) {
      console.log(error);
      toast.error("Erreur", {
        // Correction de toast.success en toast.error
        description: error.response?.data?.message || "Une erreur est survenue",
        position: "top-center",
        style: {
          backgroundColor: "#E11409FF",
          color: "white",
        },
      });
    }
  };

  // Set method to update
  const onUpdate = async (data) => {
    try {
      // Formatage des données pour l'API
      const formattedData = {
        first_name: data.first_name,
        last_name: data.last_name,
        email: data.email,
        phone: data.phone,
        subjects: data.subjects.map((subject) => subject.value),
      };
      // console.log(data);
      // console.log(rowData?.id);

      // Query
      const query = await instance.put("teachers/"+ rowData?.id, formattedData);

      // update data in state
      updateOperation(query.data.response);

      // Success response
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });

      // Ferme le modal après soumission réussie
      closeModal();
    } catch (error) {
      toast.error("Erreur", {
        // Correction de toast.success en toast.error
        description: error.response?.data?.message || "Une erreur est survenue",
        position: "top-center",
        style: {
          backgroundColor: "#E11409FF",
          color: "white",
        },
      });
    }
  };

  // Return
  return (
    <form
      onSubmit={rowData?.id ? handleSubmit(onUpdate) : handleSubmit(onSubmit)}
    >
      <div className="flex gap-2">
        <div className="">
          <Label htmlFor="first_name" className="text-right mb-2">
            Prénom(s)
          </Label>
          <input
            id="first_name"
            {...register("first_name")}
            placeholder="Prénom(s)..."
            className="px-4 py-3.5 bg-white w-full text-sm border-2 border-gray-200 focus:border-[#fb923d] rounded-md outline-none text-black"
          />
          <span className="text-red-500 text-sm">
            {errors.first_name?.message}
          </span>
        </div>

        <div className="">
          <Label htmlFor="last_name" className="text-right mb-2">
            Nom
          </Label>
          <input
            id="last_name"
            {...register("last_name")}
            placeholder="Nom..."
            className="px-4 py-3.5 bg-white w-full text-sm border-2 border-gray-200 focus:border-[#fb923d] rounded-md outline-none text-black"
          />
          <span className="text-red-500 text-sm">
            {errors.last_name?.message}
          </span>
        </div>
      </div>

      <div className="flex gap-2 py-4">
        <div className="">
          <Label htmlFor="email" className="text-right mb-2">
            Email
          </Label>
          <input
            id="email"
            type="email"
            {...register("email")}
            placeholder="Email..."
            className="px-4 py-3.5 bg-white w-full text-sm border-2 border-gray-200 focus:border-[#fb923d] rounded-md outline-none text-black"
          />
          <span className="text-red-500 text-sm">{errors.email?.message}</span>
        </div>

        <div className="">
          <Label htmlFor="phone" className="text-right mb-2">
            Contact
          </Label>
          <input
            id="phone"
            {...register("phone")}
            placeholder="Contact..."
            className="px-4 py-3.5 bg-white w-full text-sm border-2 border-gray-200 focus:border-[#fb923d] rounded-md outline-none text-black"
          />
          <span className="text-red-500 text-sm">{errors.phone?.message}</span>
        </div>
      </div>

      <div className="">
        <Label htmlFor="subjects" className="text-right mb-2">
          Matières
        </Label>
        <Controller
          name="subjects"
          control={control}
          render={({ field }) => (
            <Select
              options={levelTeacher}
              isMulti
              value={field.value}
              onChange={(options) => field.onChange(options)}
              placeholder="Choisir un niveau"
              classNamePrefix="select"
            />
          )}
        />
        <span className="text-red-500 text-sm">{errors.subjects?.message}</span>
      </div>

      <div className="flex justify-end mt-4">
        <Button type="submit" disabled={isSubmitting}>
          {isSubmitting ? (
            <Loader2 className="animate-spin w-5 h-5 mr-2" />
          ) : (
            "Enregistrer"
          )}
        </Button>
      </div>
    </form>
  );
};
