import { useState } from "react";
import { Trash } from "lucide-react";
import {
  Dialog,
  DialogTrigger,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
  DialogClose,
} from "../../../../components/ui/dialog";
import { Button } from "../../../../components/ui/button";
import instance from "../../../../lib/axiosInstance";
import { toast } from "sonner";
import { useCrudOperation } from "../../../context/CrudOperationContect";

/**
 * Composant de dialogue de suppression pour les éducateurs
 * @param {string} rowID - L'identifiant de l'éducateur à supprimer
 */
export default function DeleteDialog({ rowID }) {
  // États pour gérer le processus de suppression et l'ouverture du modal
  const [isDeleting, setIsDeleting] = useState(false);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const { deleteOperation } = useCrudOperation();

  /**
   * Gère la suppression d'un éducateur
   * Envoie une requête DELETE à l'API et met à jour l'interface utilisateur
   */
  const handleDelete = async () => {
    setIsDeleting(true);
    try {
      // Appel à l'API pour supprimer l'éducateur
      const query = await instance.delete("educators/" + rowID);
      // Mise à jour du contexte CRUD
      deleteOperation(rowID);
      // Notification de succès
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });
      // Fermeture du modal après la suppression réussie
      setIsModalOpen(false);
      setIsDeleting(false);
    } catch (error) {
      // Gestion des erreurs avec notification
      toast.success("Oops!", {
        description: error.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#E11409FF",
          color: "white",
        },
      });
    }
  };

  return (
    <Dialog open={isModalOpen} onOpenChange={setIsModalOpen}>
      {/* Bouton de déclenchement du dialogue */}
      <DialogTrigger asChild>
        <Button variant="link" className="hover:cursor-pointer">
          <Trash className="text-red-600" />
        </Button>
      </DialogTrigger>
      {/* Contenu du dialogue de confirmation */}
      <DialogContent>
        <DialogHeader className="flex flex-col gap-5">
          <DialogTitle className="uppercase">
            Confirmation de suppression
          </DialogTitle>
          <DialogDescription>
            Êtes-vous sûr de vouloir supprimer cette ligne ? <br /> Cette action
            est irréversible.
          </DialogDescription>
        </DialogHeader>
        {/* Boutons d'action */}
        <DialogFooter className="mt-4">
          <DialogClose asChild>
            <Button variant="outline">Annuler</Button>
          </DialogClose>
          <Button
            variant="destructive"
            onClick={handleDelete}
            disabled={isDeleting}
          >
            {isDeleting ? "Suppression..." : "Supprimer"}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
