"use client";
import React, { useState } from "react"; // Ajout de React
import {
  Dialog,
  DialogTrigger,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
} from "../../../../components/ui/dialog";
import { Button } from "../../../../components/ui/button";
import { PlusCircleIcon, Pencil } from "lucide-react";

/**
 * Composant FormDialog - Un dialogue modal réutilisable pour l'ajout et la modification d'éléments
 * @param {string} mode - Mode du dialogue ('add' ou 'edit')
 * @param {string} triggerLabel - Texte du bouton déclencheur
 * @param {string} triggerIcon - Icône du bouton ('plus' ou 'edit')
 * @param {string} title - Titre du dialogue
 * @param {string} description - Description du dialogue
 * @param {React.ReactElement} children - Contenu du formulaire à afficher dans le dialogue
 */
export const FormDialog = ({
  mode = "add",
  triggerLabel = "Ajouter",
  triggerIcon = "plus",
  title,
  description,
  children,
}) => {
  // État pour contrôler l'ouverture/fermeture du modal
  const [isModalOpen, setIsModalOpen] = useState(false);
  const isEdit = mode === "edit";

  /**
   * Fonction pour rendre l'icône appropriée en fonction du type
   * @returns {JSX.Element|null} L'icône correspondante ou null
   */
  const renderIcon = () => {
    if (triggerIcon === "plus") return <PlusCircleIcon className="ml-2" />;
    if (triggerIcon === "edit") return <Pencil className="ml-2" />;
    return null;
  };

  // Fonction pour fermer le modal
  const closeModal = () => setIsModalOpen(false);

  return (
    <Dialog open={isModalOpen} onOpenChange={setIsModalOpen}>
      <DialogTrigger asChild>
        {/* Rendu conditionnel du bouton en fonction du mode */}
        {triggerIcon === "plus" ? (
           <Button className='rounded-none' onClick={() => setIsModalOpen(true)}>
           <span>{triggerLabel}</span>
           {renderIcon()}
         </Button>
        ) : (
          <Button variant="link"  className="flex items-center hover:cursor-pointer">
            {renderIcon()}
          </Button>
        )}
      </DialogTrigger>

      {/* Contenu du dialogue */}
      <DialogContent>
      <DialogHeader className="border-b border-gray-200 pb-4 mb-2">
          <DialogTitle className="uppercase font-light text-[#dd7b34]">
            {title || (isEdit ? "Modifier un éducateur" : "Ajouter un éducateur")}
          </DialogTitle>
          {/* <DialogDescription>
            {description ||
              (isEdit
                ? "Modifie les informations existantes."
                : "Remplis les champs pour ajouter un élément.")}
          </DialogDescription> */}
        </DialogHeader>
        {/* Injection du formulaire enfant avec la fonction de fermeture */}
        {React.cloneElement(children, { closeModal})}
      </DialogContent>
    </Dialog>
  );
};
