"use client";
// Import des dépendances nécessaires
import { useState, useEffect } from "react";
import { Button } from "../../../../components/ui/button";
import { Label } from "../../../../components/ui/label";
import { Input } from "../../../../components/ui/input";
import Select from "react-select";
import instance from "../../../../lib/axiosInstance";
import { useForm, Controller } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as yup from "yup";
import { Loader2 } from "lucide-react";
import { useCrudOperation } from "../../../context/CrudOperationContect";
import { toast } from "sonner";
import { DialogClose } from "../../../../components/ui/dialog";
import { selectStyles } from "../../../styles/selectStyles";

// Schéma de validation YUP pour les champs du formulaire
const schema = yup.object().shape({
  first_name: yup.string().required("Le prénom est requis"),
  last_name: yup.string().required("Le nom est requis"),
  email: yup.string().required("L'email est requis"),
  phone: yup.string().required("Le numéro de téléphone est requis"),
  classrooms: yup.array().required("Veuillez sélectionner au moins une classe"),
});

// Composant principal du formulaire
export const Form = ({ closeModal, rowData }) => {
  // État pour stocker la liste des classes disponibles
  const [classrooms, setClassrooms] = useState([]);

  // Utilisation du contexte pour les opérations CRUD
  const { addOperation, updateOperation } = useCrudOperation();

  // Formatage des classes pré-sélectionnées pour le mode édition
  const defaultSelectedLevels = rowData?.classrooms.map((classroom) => ({
    value: classroom.id,
    label: classroom.name,
  }));

  // Effet pour charger la liste des classes au montage du composant
  useEffect(() => {
    const fetchLevels = async () => {
      try {
        const response = await instance.get("classrooms");
        const rawData = response.data.response.data;
        // Formatage des données pour le composant Select
        const formattedOptions = rawData.map((classroom) => ({
          value: classroom.id.toString(),
          label: classroom.name,
        }));
        setClassrooms(formattedOptions);
      } catch (error) {
        console.error("Erreur chargement niveaux :", error);
        toast.error("Erreur", {
          description: "Impossible de charger les niveaux",
          position: "top-center",
          style: {
            backgroundColor: "#E11409FF",
            color: "white",
          },
        });
      }
    };
    fetchLevels();
  }, []);

  // Configuration du formulaire avec react-hook-form
  const {
    control,
    register,
    handleSubmit,
    formState: { errors, isSubmitting },
  } = useForm({
    resolver: yupResolver(schema),
    defaultValues: {
      first_name: rowData?.first_name,
      last_name: rowData?.last_name,
      email: rowData?.email,
      phone: rowData?.phone,
      classrooms: defaultSelectedLevels,
    },
  });

  // Fonction de soumission pour l'ajout d'un nouvel éducateur
  const onSubmit = async (data) => {
    try {
      // Formatage des données pour l'API
      const formattedData = {
        first_name: data.first_name,
        last_name: data.last_name,
        email: data.email,
        phone: data.phone,
        classrooms: data.classrooms.map((classroom) => classroom.value),
      };
      const query = await instance.post("educators", formattedData);
      // Mise à jour du state avec les nouvelles données
      addOperation(query.data.response);
      // Notification de succès
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });

      closeModal();
    } catch (error) {
      // Gestion des erreurs de validation
      const errors = error.data?.response;
      if (errors) {
        Object.entries(errors).map(([field, messages]) => {
          toast.error("Erreur", {
            description: messages,
            position: "top-center",
            style: {
              backgroundColor: "#E11409FF",
              color: "white",
            },
          });
        });
      }
    }
  };

  // Fonction de mise à jour d'un éducateur existant
  const onUpdate = async (data) => {
    try {
      // Formatage des données pour l'API
      const formattedData = {
        first_name: data.first_name,
        last_name: data.last_name,
        email: data.email,
        phone: data.phone,
        classrooms: data.classrooms.map((classroom) => classroom.value),
      };

      // Requête de mise à jour
      const query = await instance.put(
        "educators/" + rowData?.id,
        formattedData
      );

      // Mise à jour des données dans le state
      updateOperation(query.data.response);

      // Notification de succès
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });

      closeModal();
    } catch (error) {
      // Gestion des erreurs de validation
      const errors = error.data?.response;
      if (errors) {
        Object.entries(errors).map(([field, messages]) => {
          toast.error("Erreur", {
            description: messages,
            position: "top-center",
            style: {
              backgroundColor: "#E11409FF",
              color: "white",
            },
          });
        });
      }
    }
  };

  // Rendu du formulaire
  return (
    <form
      onSubmit={rowData?.id ? handleSubmit(onUpdate) : handleSubmit(onSubmit)}
    >
      {/* Section des informations personnelles */}
      <div className="flex gap-2">
        <div className="">
          <Label htmlFor="first_name" className="text-right mb-2">
            Prénom(s) <span className="text-red-500">*</span>
          </Label>
          <Input
            id="first_name"
            {...register("first_name")}
            placeholder="Prénom(s)..."
          />
          <span className="text-red-500 text-sm">
            {errors.first_name?.message}
          </span>
        </div>

        <div className="">
          <Label htmlFor="last_name" className="text-right mb-2">
            Nom <span className="text-red-500">*</span>
          </Label>
          <Input
            id="last_name"
            {...register("last_name")}
            placeholder="Nom..."
          />
          <span className="text-red-500 text-sm">
            {errors.last_name?.message}
          </span>
        </div>
      </div>

      {/* Section des coordonnées */}
      <div className="flex gap-2 py-4">
        <div className="">
          <Label htmlFor="email" className="text-right mb-2">
            Email <span className="text-red-500">*</span>
          </Label>
          <Input
            id="email"
            type="email"
            {...register("email")}
            placeholder="Email..."
          />
          <span className="text-red-500 text-sm">{errors.email?.message}</span>
        </div>

        <div className="">
          <Label htmlFor="phone" className="text-right mb-2">
            Contact <span className="text-red-500">*</span>
          </Label>
          <Input
            id="phone"
            {...register("phone")}
            placeholder="Contact..."
          />
          <span className="text-red-500 text-sm">{errors.phone?.message}</span>
        </div>
      </div>

      {/* Section de sélection des classes */}
      <div className="">
        <Label htmlFor="classrooms" className="text-right mb-2">
          Classe <span className="text-red-500">*</span>
        </Label>
        <Controller
          name="classrooms"
          control={control}
          render={({ field }) => (
            <Select
              options={classrooms}
              isMulti
              value={field.value}
              onChange={(options) => field.onChange(options)}
              placeholder="Choisir un niveau"
              classNamePrefix="select"
              styles={selectStyles}
            />
          )}
        />
        <span className="text-red-500 text-sm">
          {errors.classrooms?.message}
        </span>
      </div>

      {/* Bouton de soumission */}
      <div className="flex justify-center items-center mt-4 gap-2">
        <DialogClose asChild>
          <Button variant="destructive" className="h-12">
            Annuler
          </Button>
        </DialogClose>
        <Button type="submit" disabled={isSubmitting} className="h-12">
          {isSubmitting ? (
            <Loader2 className="animate-spin w-5 h-5 mr-2" />
          ) : (
            "Enregistrer"
          )}
        </Button>
      </div>
    </form>
  );
};
