"use client";
// Start import
import { useState, useEffect } from "react";
import { Button } from "../../../../components/ui/button";
import { Label } from "../../../../components/ui/label";
import { Input } from "../../../../components/ui/input";
import Select from "react-select";
import instance from "../../../../lib/axiosInstance";
import { useForm, Controller } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as yup from "yup";
import { Loader2 } from "lucide-react";
import { DialogClose } from "../../../../components/ui/dialog";
import { selectStyles } from "../../../styles/selectStyles";
import { useCrudOperation } from "../../../context/CrudOperationContect";
import { toast } from "sonner";
// End import

// Schéma de validation YUP pour le formulaire des enseignants
// Définit les règles de validation pour chaque champ du formulaire
const schema = yup.object().shape({
  first_name: yup.string().required("Le prénom est obligatoire"),
  last_name: yup.string().required("Le nom est obligatoire"),
  email: yup.string().required("L'email est obligatoire").email("Format d'email invalide"),
  phone: yup.string().required("Le numéro de téléphone est obligatoire"),
  subjects: yup.array().required("Veuillez sélectionner au moins une matière"),
});

// Composant Form pour la création et la modification des enseignants
// Props:
// - closeModal: fonction pour fermer le modal
// - rowData: données de l'enseignant pour la modification
export const Form = ({ closeModal, rowData }) => {
  // État pour stocker les options des matières
  const [levelTeacher, setTeacher] = useState([]);

  // Utilisation du contexte pour les opérations CRUD
  const { addOperation, updateOperation } = useCrudOperation();

  // console.log(rowData);

  // Formatage des matières pré-sélectionnées pour le mode édition
  const defaultSelectedLevels = rowData?.subjects.map((subject) => ({
    value: subject.id,
    label: subject.name,
  }));

  // Effet pour charger les matières depuis l'API
  useEffect(() => {
    const fetchLevelOptions = async () => {
      try {
        const response = await instance.get("settings/subjects");
        const rawData = response.data.response.data;
        // Formatage des données pour le composant Select
        const formattedOptions = rawData.map((subject) => ({
          value: subject.id.toString(),
          label: subject.name,
        }));
        setTeacher(formattedOptions);
      } catch (error) {
        console.error("Erreur chargement niveaux :", error);
        toast.error("Erreur", {
          description: "Impossible de charger les niveaux",
          position: "top-center",
        });
      }
    };
    fetchLevelOptions();
  }, []);

  // Configuration du formulaire avec react-hook-form
  const {
    control,
    register,
    handleSubmit,
    formState: { errors, isSubmitting },
  } = useForm({
    resolver: yupResolver(schema),
    defaultValues: {
      first_name: rowData?.first_name,
      last_name: rowData?.last_name,
      email: rowData?.email,
      phone: rowData?.phone,
      subjects: defaultSelectedLevels,
    },
  });

  // Fonction pour créer un nouvel enseignant
  const onSubmit = async (data) => {
    try {
      // Formatage des données pour l'API
      const formattedData = {
        first_name: data.first_name,
        last_name: data.last_name,
        email: data.email,
        phone: data.phone,
        subjects: data.subjects.map((subject) => subject.value),
      };
      const query = await instance.post("teachers", formattedData);
      addOperation(query.data.response);
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });

      // Ferme le modal après soumission réussie
      closeModal();
    } catch (error) {
      // Gestion des erreurs de validation
      const errors = error.data?.response;
      if (errors) {
        Object.entries(errors).map(([field, messages]) => {
          toast.error("Erreur", {
            description: messages,
            position: "top-center",
            style: {
              backgroundColor: "#E11409FF",
              color: "white",
            },
          });
        });
      }
    }
  };

  // Fonction pour modifier un enseignant existant
  const onUpdate = async (data) => {
    try {
      // Formatage des données pour l'API
      const formattedData = {
        first_name: data.first_name,
        last_name: data.last_name,
        email: data.email,
        phone: data.phone,
        subjects: data.subjects.map((subject) => subject.value),
      };

      // Query
      const query = await instance.put("teachers/"+ rowData?.id, formattedData);

      // update data in state
      updateOperation(query.data.response);

      // Success response
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });

      // Ferme le modal après soumission réussie
      closeModal();
    } catch (error) {
      // Gestion des erreurs de validation
      const errors = error.data?.response;
      if (errors) {
        Object.entries(errors).map(([field, messages]) => {
          toast.error("Erreur", {
            description: messages,
            position: "top-center",
            style: {
              backgroundColor: "#E11409FF",
              color: "white",
            },
          });
        });
      }
    }
  };

  // Rendu du formulaire
  return (
    <form
      onSubmit={rowData?.id ? handleSubmit(onUpdate) : handleSubmit(onSubmit)}
    >
      {/* Section des informations personnelles */}
      <div className="flex gap-2">
        {/* Champ Prénom */}
        <div className="">
          <Label htmlFor="first_name" className="text-right mb-2">
            Prénom(s) <span className="text-red-500">*</span>
          </Label>
          <Input
            id="first_name"
            {...register("first_name")}
            placeholder="Prénom(s)..."
          />
          <span className="text-red-500 text-sm">
            {errors.first_name?.message}
          </span>
        </div>

        {/* Champ Nom */}
        <div className="">
          <Label htmlFor="last_name" className="text-right mb-2">
            Nom <span className="text-red-500">*</span>
          </Label>
          <Input
            id="last_name"
            {...register("last_name")}
            placeholder="Nom..."
          />
          <span className="text-red-500 text-sm">
            {errors.last_name?.message}
          </span>
        </div>
      </div>

      {/* Section des coordonnées */}
      <div className="flex gap-2 py-4">
        {/* Champ Email */}
        <div className="">
          <Label htmlFor="email" className="text-right mb-2">
            Email <span className="text-red-500">*</span>
          </Label>
          <Input
            id="email"
            type="email"
            {...register("email")}
            placeholder="Email..."
          />
          <span className="text-red-500 text-sm">{errors.email?.message}</span>
        </div>

        {/* Champ Contact */}
        <div className="">
          <Label htmlFor="phone" className="text-right mb-2">
            Contact <span className="text-red-500">*</span>
          </Label>
          <Input
            id="phone"
            {...register("phone")}
            placeholder="Contact..."
          />
          <span className="text-red-500 text-sm">{errors.phone?.message}</span>
        </div>
      </div>

      {/* Section des matières */}
      <div className="">
        <Label htmlFor="subjects" className="text-right mb-2">
          Matières <span className="text-red-500">*</span>
        </Label>
        <Controller
          name="subjects"
          control={control}
          render={({ field }) => (
            <Select
              options={levelTeacher}
              isMulti
              value={field.value}
              onChange={(options) => field.onChange(options)}
              placeholder="Choisir une matière"
              classNamePrefix="select"
              styles={selectStyles}
            />
          )}
        />
        <span className="text-red-500 text-sm">{errors.subjects?.message}</span>
      </div>

      {/* Bouton de soumission */}
      <div className="flex justify-center items-center mt-4 gap-2">
        <DialogClose asChild>
          <Button variant="destructive" className="h-12">
            Annuler
          </Button>
        </DialogClose>
        <Button type="submit" disabled={isSubmitting} className="h-12">
          {isSubmitting ? (
            <Loader2 className="animate-spin w-5 h-5 mr-2" />
          ) : (
            "Enregistrer"
          )}
        </Button>
      </div>
    </form>
  );
};
