"use client";

import { useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as yup from "yup";
import { Loader2 } from "lucide-react";
import { useEffect, useState } from "react";
import Image from "next/image";
import profilePic from "../../../public/assets/profile/empty-classroom.jpg";
import { IoIosSchool } from "react-icons/io";
import { FaLongArrowAltRight } from "react-icons/fa";
import Link from "next/link";
import { toast } from "sonner";
import { Button } from "../../../components/ui/button";
import instance from "../../../lib/axiosInstance";
import {
  UserCog,
  Mail,
  Phone,
  MapPin,
  User,
  CalendarDays,
  KeyRound,
  UserRoundPen,
} from "lucide-react";

import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "../../../components/ui/dialog";

// Statistiques du profil
const stats = [
  { data: "35K", title: "Customers" },
  { data: "10K+", title: "Downloads" },
  { data: "40+", title: "Countries" },
  { data: "30M+", title: "Total revenue" },
  { data: "35K", title: "Customers" },
  { data: "10K+", title: "Downloads" },
];

const schema = yup
  .object({
    email: yup.string().required(),
    phone: yup.string().required(),
    first_name: yup.string().required(),
    last_name: yup.string().required(),
  })
  .required();

export default function Profil() {
  const [user, setUser] = useState(null);
  // Set state for dialog epening
  const [open, setOpen] = useState(false);

  useEffect(() => {
    const profile = async () => {
      try {
        const query = await instance.get("profile");
        const profileData = query.data.response;
        setUser(profileData);
      } catch (error) {
        console.error("Erreur chargement niveaux :", error);
      }
    };
    profile();
  }, []);

  const {
    register,
    handleSubmit,
    reset,
    formState: { errors, isSubmitting },
  } = useForm({
    resolver: yupResolver(schema),
    defaultValues: {
      email: "",
      phone: "",
      last_name: "",
      first_name: "",
    },
  });

  useEffect(() => {
    if (open && user) {
      reset({
        email: user?.email,
        phone: user?.phone,
        last_name: user?.last_name,
        first_name: user?.first_name,
      });
    }
  }, [open, user, reset]);

  const onSubmit = async (data) => {
    try {
      const query = await instance.put("profile", data);
      setUser((prevUser) => ({ ...prevUser, ...data }));
      setOpen(false);
      toast.success("Modifié", {
        description: query.data.message,
        position: "top-center",
      });
    } catch (error) {
      const response = error.data;
      toast.error(response.message, {
        description: response.response.error,
        position: "top-center",
      });
    }
  };

  return (
    <div className="container mx-auto">
      <div className="grid grid-cols-4 gap-10">

        <div className="flex flex-col gap-2">
          <div className="shadow-xl p-5">
            <div className="flex items-center gap-4 flex-col">
              <div className="bg-gray-100 p-3 rounded-full">
                <User className="text-[#032313]" size={32} />
              </div>
              <div>
                <h2 className="text-xl font-semibold text-gray-800">
                  {user?.first_name}
                </h2>
                <div className="flex gap-5">
                  <p className="text-sm text-gray-500">{user?.email}</p>
                  <Dialog open={open} onOpenChange={setOpen}>
                    <DialogTrigger asChild>
                      <Button variant="outline">
                        <UserRoundPen className="text-[#611c14]" />
                      </Button>
                    </DialogTrigger>
                    <DialogContent className="sm:max-w-[425px]">
                      <DialogHeader>
                        <DialogTitle className="uppercase">Profil</DialogTitle>
                        <DialogDescription></DialogDescription>
                      </DialogHeader>
                      <form onSubmit={handleSubmit(onSubmit)}>
                        <div className="grid gap-4 py-4">
                          <div className="flex flex-row gap-2">
                            <div className="relative flex items-center flex-col">
                              <label
                                htmlFor="first_name"
                                className="text-gray-800 text-[13px] bg-white absolute px-2 top-[-9px] left-[18px] font-semibold"
                              >
                                Prénoms
                              </label>
                              <input
                                id="first_name"
                                {...register("first_name")}
                                placeholder="Prénoms..."
                                className="px-4 py-3.5 bg-white w-full text-sm border-2 border-gray-200 focus:border-[#fb923d] rounded-md outline-none text-black"
                              />
                              <p className="text-red-500">
                                {errors.first_name?.message}
                              </p>
                            </div>
                            <div className="relative flex items-center flex flex-col">
                              <label
                                htmlFor="last_name"
                                className="text-gray-800 text-[13px] bg-white absolute px-2 top-[-9px] left-[18px] font-semibold"
                              >
                                Nom
                              </label>
                              <input
                                id="last_name"
                                {...register("last_name")}
                                placeholder="Nom..."
                                className="px-4 py-3.5 bg-white w-full text-sm border-2 border-gray-200 focus:border-[#fb923d] rounded-md outline-none text-black"
                              />
                              <p className="text-red-500">
                                {errors.last_name?.message}
                              </p>
                            </div>
                          </div>
                          <div className="relative flex items-center flex-col">
                            <label
                              htmlFor="email"
                              className="text-gray-800 text-[13px] bg-white absolute px-2 top-[-9px] left-[18px] font-semibold"
                            >
                              Email
                            </label>
                            <input
                              id="email"
                              {...register("email")}
                              type="email"
                              placeholder="Email..."
                              className="px-4 py-3.5 bg-white w-full text-sm border-2 border-gray-200 focus:border-[#fb923d] rounded-md outline-none text-black"
                            />
                            <p className="text-red-500">
                              {errors.email?.message}
                            </p>
                          </div>
                          <div className="relative flex items-center flex-col">
                            <label
                              htmlFor="phone"
                              className="text-gray-800 text-[13px] bg-white absolute px-2 top-[-9px] left-[18px] font-semibold"
                            >
                              Phone
                            </label>
                            <input
                              id="phone"
                              {...register("phone")}
                              placeholder="Téléphone..."
                              className="px-4 py-3.5 bg-white w-full text-sm border-2 border-gray-200 focus:border-[#fb923d] rounded-md outline-none text-black"
                            />
                            <p className="text-red-500">
                              {errors.phone?.message}
                            </p>
                          </div>

                          <div className="mt-1">
                            <Button
                              type="submit"
                              disabled={isSubmitting}
                              className="w-full h-12 uppercase bg-[#611c14]"
                            >
                              {isSubmitting ? (
                                <Loader2 className="animate-spin w-5 h-5 mr-2" />
                              ) : (
                                "Enregistrer"
                              )}
                            </Button>
                          </div>
                        </div>
                      </form>
                    </DialogContent>
                  </Dialog>
                </div>
              </div>
            </div>

            <div className="mt-5">
              <h1 className="uppercase text-center">Contacts</h1>
              <div className="space-y-2 text-sm text-gray-700 flex flex-col gap-2 mt-2">
                <div className="flex items-center gap-2 ">
                  <Phone className="text-[#032313]" size={18} />
                  <span>{user?.phone}</span>
                </div>
                <div className="flex items-center gap-2 ">
                  <Mail className="text-[#032313]" size={18} />
                  <span>{user?.email}</span>
                </div>
              </div>
            </div>
          </div>

          <div className="shadow-xl p-5">
            <div>
              <h1 className="uppercase text-center">Établissement</h1>
              <div className="space-y-2 text-sm text-gray-700 flex flex-col gap-2 mt-5">
                <div className="flex items-center gap-2 ">
                  <Phone className="text-[#032313]" size={18} />
                  <span>{user?.school?.name}</span>
                </div>
                <div className="flex items-center gap-2">
                  <Phone className="text-[#032313]" size={18} />
                  <span>{user?.school?.phone}</span>
                </div>
                <div className="flex items-center gap-2">
                  <MapPin className="text-[#032313]" size={18} />
                  <span>{user?.school?.address}</span>
                </div>
                <div className="flex items-center gap-2">
                  <KeyRound className="text-[#032313]" size={18} />
                  <span>{user?.school?.code}</span>
                </div>
                <div className="flex items-center gap-2">
                  <CalendarDays className="text-[#032313]" size={18} />
                  <span>Crée le {user?.school?.created_at}</span>
                </div>
              </div>
            </div>
          </div>

        </div>

        
        <div className="col-span-3">
          <div className="h-[270px]">
            <Image
              src={profilePic}
              alt="Photo de profil"
              className="object-cover object-center w-full h-full rounded-xl"
            />
          </div>

          <div className="grid grid-cols-4 gap-5 mt-5">
            <div className="flex flex-col gap-4 border border-[#fb923d] bg-[#fb923d] rounded-xl p-5">
              <div className="flex items-center gap-3">
                <IoIosSchool className="text-4xl text-white" />
                <div className="flex flex-col gap-0">
                  <h1 className="text-xl font-semibold text-black">325</h1>
                  <span className="text-gray font-bold text-xl">Élèves</span>
                </div>
              </div>
              <hr className="border-[#611c14]" />
              <div>
                <div className="flex flex-row justify-between">
                  <span>
                    <Link href="dashboard" className="text-white">
                      Consulter la lsite
                    </Link>
                  </span>
                  <Link href="dashboard">
                    <FaLongArrowAltRight className="text-xl text-white" />
                  </Link>
                </div>
              </div>
            </div>

            <div className="flex flex-col gap-4 border border-[#611c14] rounded-xl p-5">
              <div className="flex items-center gap-3">
                <IoIosSchool className="text-4xl text-[#611c14]" />
                <div className="flex flex-col gap-0">
                  <h1 className="text-xl font-semibold text-black">10</h1>
                  <span className="text-gray font-bold text-xl">Classe</span>
                </div>
              </div>
              <hr className="border-[#611c14]" />
              <div>
                <div className="flex flex-row justify-between">
                  <span>
                    <Link href="dashboard">Consulter la lsite</Link>
                  </span>
                  <Link href="dashboard">
                    <FaLongArrowAltRight className="text-xl text-[#032313]" />
                  </Link>
                </div>
              </div>
            </div>

            <div className="flex flex-col gap-4 border border-[#fb923d] bg-[#fb923d] rounded-xl p-5">
              <div className="flex items-center gap-3">
                <IoIosSchool className="text-4xl text-white" />
                <div className="flex flex-col gap-0">
                  <h1 className="text-xl font-semibold text-black">25</h1>
                  <span className="text-gray font-bold text-xl">Matières</span>
                </div>
              </div>
              <hr className="border-[#611c14]" />
              <div>
                <div className="flex flex-row justify-between">
                  <span>
                    <Link href="dashboard" className="text-white">
                      Consulter la lsite
                    </Link>
                  </span>
                  <Link href="dashboard">
                    <FaLongArrowAltRight className="text-xl text-white" />
                  </Link>
                </div>
              </div>
            </div>

            <div className="flex flex-col gap-4 border border-[#611c14] rounded-xl p-5">
              <div className="flex items-center gap-3">
                <IoIosSchool className="text-4xl text-[#611c14]" />
                <div className="flex flex-col gap-0">
                  <h1 className="text-xl font-semibold text-black">55</h1>
                  <span className="text-gray font-bold text-xl">
                    Enseignants
                  </span>
                </div>
              </div>
              <hr className="border-[#611c14]" />
              <div>
                <div className="flex flex-row justify-between">
                  <span>
                    <Link href="dashboard">Consulter la lsite</Link>
                  </span>
                  <Link href="dashboard">
                    <FaLongArrowAltRight className="text-xl text-[#032313]" />
                  </Link>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
