"use client";
// Start import
import { useState, useEffect } from "react";
import { Button } from "../../../../components/ui/button";
import { Label } from "../../../../components/ui/label";
import { Input } from "../../../../components/ui/input";
import Select from "react-select";
import instance from "../../../../lib/axiosInstance";
import { useForm, Controller } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import { DialogClose } from "../../../../components/ui/dialog";
import { selectStyles } from "../../../styles/selectStyles";
import * as yup from "yup";
import { Loader2 } from "lucide-react";
import { useCrudOperation } from "../../../context/CrudOperationContect";
import { toast } from "sonner";
// End import

// Schéma de validation YUP
const schema = yup.object().shape({
  name: yup.string().required("Le nom de la classe est requis"),
  description: yup.string().nullable(),
  level_id: yup.object().required("Le niveau est requis"),
});

// Export
export const Form = ({ closeModal, rowData }) => {
  // Set state for levels
  const [levels, setLevels] = useState([]);

  // Utilisation du contexte pour accéder aux données et fonctions
  const { addOperation, updateOperation } = useCrudOperation();

  // Set useEffect hook
  useEffect(() => {
    const fetchLevelOptions = async () => {
      try {
        const response = await instance.get("settings/levels");
        const rawData = response.data.response.data;
        const formattedOptions = rawData.map((level) => ({
          value: level.id.toString(),
          label: level.name,
        }));
        setLevels(formattedOptions);
      } catch (error) {
        console.error("Erreur chargement niveaux :", error);
        toast.error("Erreur", {
          description: "Impossible de charger les niveaux",
          position: "top-center",
        });
      }
    };
    fetchLevelOptions();
  }, []);

  // Format pre-selected data
  const defaultSelectedLevels = rowData?.level ? {
    value: rowData.level.id.toString(),
    label: rowData.level.name,
  } : null;

  // Define react-hook-form
  const {
    control,
    register,
    handleSubmit,
    formState: { errors, isSubmitting },
  } = useForm({
    resolver: yupResolver(schema),
    defaultValues: {
      description: rowData?.description,
      name: rowData?.name,
      level_id: defaultSelectedLevels,
    },
  });

  // Set method to add classrooms
  const onSubmit = async (data) => {
    try {
      // Formatage des données pour l'API
      const formattedData = {
        name: data.name,
        description: data.description,
        level_id: data.level_id.value,
      };

      // console.log(data);
      const query = await instance.post("classrooms", formattedData);
      addOperation(query.data.response);
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });

      // Ferme le modal après soumission réussie
      closeModal();
    } catch (error) {
      console.log(error.data?.message);
      console.log(error.data?.response);
      const errors = error.data?.response;
      if (errors) {
        errors.name.forEach((item) => {
          toast.error("Erreur", {
            description: item,
            position: "top-center",
            style: {
              backgroundColor: "#E11409FF",
              color: "white",
            },
          });
        });
      }
    }
  };

  // Set method to update
  const onUpdate = async (data) => {
    try {
      // Formatage des données pour l'API
      const formattedData = {
        name: data.name,
        description: data.description,
        level_id: data.level_id.value,
      };

      // Set query ti update
      const query = await instance.put(
        "classrooms/" + rowData?.id,
        formattedData
      );

      // Update to state
      updateOperation(query.data.response);

      // Display response
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });

      // Ferme le modal après soumission réussie
      closeModal();
    } catch (error) {
      console.log(error);
      toast.error("Erreur", {
        // Correction de toast.success en toast.error
        description: error.response?.data?.message || "Une erreur est survenue",
        position: "top-center",
        style: {
          backgroundColor: "#E11409FF",
          color: "white",
        },
      });
    }
  };

  // Return
  return (
    <form
      onSubmit={rowData?.id ? handleSubmit(onUpdate) : handleSubmit(onSubmit)}
    >
      <div className="">
        <Label htmlFor="name" className="text-right mb-2">
          Classe <span className="text-red-500">*</span>
        </Label>
        <Input
          id="name"
          {...register("name")}
          placeholder="Classe..."
          className="w-full"
        />
        <span className="text-red-500 text-sm">{errors.name?.message}</span>
      </div>

      <div className="py-4">
        <Label htmlFor="level_id" className="text-right mb-2">
          Niveau <span className="text-red-500">*</span>
        </Label>
        <Controller
          name="level_id"
          control={control}
          render={({ field }) => (
            <Select
              options={levels}
              value={field.value}
              onChange={(options) => field.onChange(options)}
              placeholder="Choisir un niveau"
              classNamePrefix="select"
              styles={selectStyles}
            />
          )}
        />
        <span className="text-red-500 text-sm">{errors.level_id?.message}</span>
      </div>

      <div className="">
        <Label htmlFor="description" className="text-right mb-2">
          Description
        </Label>
        <textarea
          id="description"
          {...register("description")}
          className="px-4 py-3.5 bg-white w-full text-sm border-2 border-gray-200 focus:border-[#fb923d] rounded-md outline-none text-black"
        ></textarea>
        <span className="text-red-500 text-sm">{errors.description?.message}</span>
      </div>

      <div className="flex justify-center items-center mt-4 gap-2">
        <DialogClose asChild>
          <Button variant="destructive" className="h-12">
            Annuler
          </Button>
        </DialogClose>
        <Button type="submit" disabled={isSubmitting} className="h-12">
          {isSubmitting ? (
            <Loader2 className="animate-spin w-5 h-5 mr-2" />
          ) : (
            "Enregistrer"
          )}
        </Button>
      </div>
    </form>
  );
};
