"use client";
// Start import
import { useState, useEffect } from "react";
import { Button } from "../../../../../components/ui/button";
import { Label } from "../../../../../components/ui/label";
import { Input } from "../../../../../components/ui/input";
import Select from "react-select";
import instance from "../../../../../lib/axiosInstance";
import { useForm, Controller } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as yup from "yup";
import { Loader2 } from "lucide-react";
import { useCrudOperation } from "../../../../context/CrudOperationContect";
import { toast } from "sonner";
import DatePicker from "react-datepicker";
import "react-datepicker/dist/react-datepicker.css";
import { registerLocale } from "react-datepicker";
import fr from "date-fns/locale/fr";
registerLocale("fr", fr);
import Link from "next/link";
import { selectStyles } from "../../../../styles/selectStyles";


// End import




// Schéma de validation YUP pour le formulaire de contrôle
// Définit les règles de validation pour chaque champ du formulaire
const schema = yup.object().shape({
  name: yup.string().required("Le nom de l'examen est requis"),
  session: yup.number().typeError("Le nombre de session doit être un nombre"),
  total_points: yup.number().typeError("Le nombre total de points doit être un nombre"),
  pass_points: yup.number().required("Les points requis sont obligatoires").typeError("Les points requis doivent être un nombre"),
  start_date: yup.date().required("La date de début est requise").typeError("La date de début doit être une date valide"),
  end_date: yup.date().required("La date de fin est requise").typeError("La date de fin doit être une date valide"),
  level: yup.object().nullable(),
  file: yup.mixed(),
  description: yup.string(),
});



export default function Form({ examID = null }) {
  // États pour gérer les données du formulaire
  const [levels, setLevels] = useState([]); // Liste des matières disponibles
  const [studentData, setStudentData] = useState(null); // Données du contrôle si en mode édition

  // Contexte pour les opérations CRUD
  const { addOperation, updateOperation } = useCrudOperation();

  // Fonction pour récupérer la liste des matières de la classe
  const fetchSubjects = async () => {
    try {
      const response = await instance.get(`settings/levels`);
      const rawData = response.data.response.data;
      const formattedOptions = rawData.map((data) => ({
        value: data.id.toString(),
        label: data.name,
      }));
      setLevels(formattedOptions);
    } catch (error) {
      console.error("Erreur chargement niveaux :", error);
    }
  };

  // Fonction pour récupérer les données d'un contrôle existant
  const fetchStudentByID = async () => {
    try {
      const response = await instance.get(
        `exams/${examID}`
      );
      setStudentData(response.data.response);
    } catch (error) {
      console.error("Erreur chargement du contrôle :", error);
    }
  };

  // Chargement initial des données
  useEffect(() => {
    fetchSubjects();
    if (examID) {
      fetchStudentByID();
    }
  }, [examID]);

  // Configuration du formulaire avec react-hook-form
  const {
    control,
    register,
    handleSubmit,
    formState: { errors, isSubmitting },
    reset,
  } = useForm({
    defaultValues: {
      name: "",
      sessions: "",
      total_points: "",
      pass_points: "",
      start_date: null,
      end_date: null,
      level: null,
      file: null,
      description: ""
    },
    resolver: yupResolver(schema),
  });

  // Mise à jour des valeurs du formulaire quand les données de l'examen sont chargées
  useEffect(() => {
    if (studentData) {
      console.log(studentData);
      reset({
        name: studentData.name,
        session: studentData.sessions,
        total_points: studentData.total_points,
        pass_points: studentData.pass_points,
        start_date: studentData.start_date ? new Date(studentData.start_date) : null,
        end_date: studentData.end_date ? new Date(studentData.end_date) : null,
        level: studentData.level ? {
          value: studentData.level.id.toString(),
          label: studentData.level.name
        } : null,
        file: null,
        description: studentData.description || ""
      });
    }
  }, [studentData, reset]);

  // Fonction de soumission pour la création d'un nouveau contrôle
  const onSubmit = async (data) => {
    try {
      // Formatage des données pour l'API
      const formattedData = {
        name: data.name,
        sessions: parseInt(data.session),
        total_points: data.total_points ? parseInt(data.total_points) : null,
        pass_points: parseInt(data.pass_points),
        start_date: data.start_date.toISOString().split("T")[0],
        end_date: data.end_date.toISOString().split("T")[0],
        level_id: data.level?.value || null,
        file: data.file?.[0] || null,
        description: data.description || null
      };

      // Envoi de la requête de création
      const query = await instance.post(
        `exams`,
        formattedData
      );

      // Mise à jour de l'état global
      addOperation(query.data.response);

      // Affichage du message de succès
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });
    } catch (error) {
      // Gestion des erreurs de validation
      const errors = error.data?.response;
      if (errors) {
        Object.entries(errors).map(([field, messages]) => {
          toast.error("Erreur", {
            description: messages,
            position: "top-center",
            style: {
              backgroundColor: "#E11409FF",
              color: "white",
            },
          });
        });
      }
    }
  };

  // Fonction de mise à jour pour un contrôle existant
  const onUpdate = async (data) => {
    try {
      // Formatage des données pour l'API
      const formattedData = {
        name: data.name,
        sessions: parseInt(data.session),
        total_points: data.total_points ? parseInt(data.total_points) : null,
        pass_points: parseInt(data.pass_points),
        start_date: data.start_date.toISOString().split("T")[0],
        end_date: data.end_date.toISOString().split("T")[0],
        level_id: data.level?.value || null,
        file: data.file?.[0] || null,
        description: data.description || null
      };

      // Envoi de la requête de mise à jour
      const query = await instance.put(
        `exams/${examID}`,
        formattedData
      );

      // Mise à jour de l'état global
      updateOperation(query.data.response);

      // Affichage du message de succès
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });
    } catch (error) {
      // Gestion des erreurs de validation
      const errors = error.data?.response;
      if (errors) {
        Object.entries(errors).map(([field, messages]) => {
          toast.error("Erreur", {
            description: messages,
            position: "top-center",
            style: {
              backgroundColor: "#E11409FF",
              color: "white",
            },
          });
        });
      }
    }
  };

  return (
    <form
      onSubmit={examID ? handleSubmit(onUpdate) : handleSubmit(onSubmit)}
      className="space-y-6"
    >
      <div className="space-y-6">
        <div className="space-y-2">
          <Label htmlFor="name">
            Examen <span className="text-red-500">*</span>
          </Label>
          <Input
            id="name"
            {...register("name")}
            className={`h-12 ${errors.name ? "border-red-500" : ""}`}
          />
          {errors.name && (
            <p className="text-sm text-red-600">{errors.name.message}</p>
          )}
        </div>
      </div>

      <div className="space-y-6">
        <div className="grid grid-cols-1 md:grid-cols-8 gap-6">
          <div className="space-y-2 col-span-2">
            <Label htmlFor="session">
              Session <span className="text-red-500">*</span>
            </Label>
            <Input
              id="session"
              type="number"
              {...register("session")}
              className={`h-12 ${errors.session ? "border-red-500" : ""}`}
            />
            {errors.session && (
              <p className="text-sm text-red-600">{errors.session.message}</p>
            )}
          </div>

          <div className="space-y-2 col-span-3">
            <Label htmlFor="total_points">
              Nbre total de points de l'examen
            </Label>
            <Input
              id="total_points"
              type="number"
              {...register("total_points")}
              className="h-12"
            />
            {errors.total_points && (
              <p className="text-sm text-red-600">
                {errors.total_points.message}
              </p>
            )}
          </div>

          <div className="space-y-2 col-span-3">
            <Label htmlFor="pass_points">
              Points requis pour réussir l'examen <span className="text-red-500">*</span>
            </Label>
            <Input
              id="pass_points"
              type="number"
              {...register("pass_points")}
              className={`h-12 ${errors.pass_points ? "border-red-500" : ""}`}
            />
            {errors.pass_points && (
              <p className="text-sm text-red-600">
                {errors.pass_points.message}
              </p>
            )}
          </div>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <div className="space-y-2">
          <Label htmlFor="start_date">
            Date de début de l'examen <span className="text-red-500">*</span>
          </Label>
          <Controller
            name="start_date"
            control={control}
            render={({ field }) => (
              <DatePicker
                selected={field.value}
                onChange={(date) => field.onChange(date)}
                dateFormat="dd/MM/yyyy"
                className={`w-full h-12 px-3 border rounded-md ${
                  errors.start_date ? "border-red-500" : ""
                }`}
                placeholderText="Sélectionner une date"
                locale="fr"
                minDate={new Date()}
              />
            )}
          />
          {errors.start_date && (
            <p className="text-sm text-red-600">{errors.start_date.message}</p>
          )}
        </div>

        <div className="space-y-2">
          <Label htmlFor="end_date">
            Date de fin de l'examen <span className="text-red-500">*</span>
          </Label>
          <Controller
            name="end_date"
            control={control}
            render={({ field }) => (
              <DatePicker
                selected={field.value}
                onChange={(date) => field.onChange(date)}
                dateFormat="dd/MM/yyyy"
                className={`w-full h-12 px-3 border rounded-md ${
                  errors.end_date ? "border-red-500" : ""
                }`}
                placeholderText="Sélectionner une date"
                locale="fr"
                minDate={new Date()}
              />
            )}
          />
          {errors.end_date && (
            <p className="text-sm text-red-600">{errors.end_date.message}</p>
          )}
        </div>

        <div className="space-y-2">
          <Label htmlFor="level">Niveau</Label>
          <Controller
            name="level"
            control={control}
            render={({ field }) => (
              <Select
                options={levels}
                value={field.value}
                onChange={(options) => field.onChange(options)}
                placeholder="Choisir une matière"
                classNamePrefix="select"
                styles={selectStyles}
              />
            )}
          />
          {errors.level && (
            <p className="text-sm text-red-600">{errors.level.message}</p>
          )}
        </div>
      </div>

      <div className="space-y-2">
        <Label htmlFor="file">Fichier</Label>
        <Input
          id="file"
          type="file"
          {...register("file")}
          className="h-12"
        />
      </div>

      <div className="space-y-2">
        <Label htmlFor="description">Description</Label>
        <textarea
          id="description"
          {...register("description")}
          className="w-full h-32 px-3 py-2 border rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 resize-none"
          placeholder="Entrez la description de l'examen..."
        />
      </div>

      <div className="flex gap-5 justify-center pt-6">
        <Link
          href={`/schoollife/exams`}
          className="flex items-center justify-center h-12 border text-sm bg-red-700 text-white px-5 w-40"
        >
          Retour
        </Link>

        <Button
          type="submit"
          disabled={isSubmitting}
          className="w-40 rounded-none h-12"
        >
          {isSubmitting ? (
            <Loader2 className="animate-spin w-5 h-5 mr-2" />
          ) : (
            "Enregistrer"
          )}
        </Button>
      </div>
    </form>
  );
}
