"use client";

// Start import
import { useState, useEffect } from "react";
import { Button } from "../../../../components/ui/button";
import { Label } from "../../../../components/ui/label";
import Select from "react-select";
import instance from "../../../../lib/axiosInstance";
import { useForm, Controller } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as yup from "yup";
import { Loader2 } from "lucide-react";
import { useCrudOperation } from "../../../context/CrudOperationContect";
import { toast } from "sonner";
import { DialogClose } from "../../../../components/ui/dialog";
import { selectStyles } from "../../../styles/selectStyles";


// End import

// Schéma de validation YUP
const schema = yup.object().shape({
  level_id: yup.string().required("Le niveau est obligatoire"),
});

// Export
export const Form = ({ closeModal }) => {
  // Set state for levelsOption
  const [levelOptions, setLevelOptions] = useState([]);

  // Utilisation du contexte pour accéder aux données et fonctions
  const { addOperation } = useCrudOperation();

  // Set useEffect hook
  useEffect(() => {
    const fetchLevelOptions = async () => {
      try {
        const response = await instance.get("settings/levels/options");
        const rawData = response.data.response.data;
        const formattedOptions = rawData.map((level) => ({
          value: level.id.toString(),
          label: level.name,
        }));
        setLevelOptions(formattedOptions);
      } catch (error) {
        console.error("Erreur chargement niveaux :", error);
      }
    };
    fetchLevelOptions();
  }, []);

  // Define react-hook-form
  const {
    control,
    handleSubmit,
    formState: { errors, isSubmitting },
  } = useForm({
    resolver: yupResolver(schema),
  });

  // Set method to add levels
  const onSubmit = async (data) => {
    try {
      const query = await instance.post("/settings/levels", data);
      addOperation(query.data.response);
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });

      // Ferme le modal après soumission réussie
      closeModal();
    } catch (error) {
      // Gestion des erreurs de validation
      const errors = error.data?.response;
      if (errors) {
        Object.entries(errors).map(([field, messages]) => {
          toast.error("Erreur", {
            description: messages,
            position: "top-center",
            style: {
              backgroundColor: "#E11409FF",
              color: "white",
            },
          });
        });
      }
    }
  };

  // Return
  return (
    <form onSubmit={handleSubmit(onSubmit)}>
      <div className="py-4">
        <Label htmlFor="level_id" className="text-right mb-2">
          Niveau <span className="text-red-500">*</span>
        </Label>

        <Controller
          name="level_id"
          control={control}
          render={({ field }) => (
            <Select
              options={levelOptions}
              value={levelOptions.find((opt) => opt.value === field.value)}
              onChange={(option) => field.onChange(option?.value)}
              placeholder="Choisir un niveau"
              className="w-full"
              styles={selectStyles}

            />
          )}
        />
        {errors.level_id && (
          <span className="text-red-500 text-sm mt-1">{errors.level_id.message}</span>
        )}
      </div>

      <div className="flex justify-center items-center mt-4 gap-2">
        <DialogClose asChild>
          <Button variant="destructive" className="h-12">
            Annuler
          </Button>
        </DialogClose>
        <Button type="submit" disabled={isSubmitting} className="h-12">
          {isSubmitting ? (
            <Loader2 className="animate-spin w-5 h-5 mr-2" />
          ) : (
            "Enregistrer"
          )}
        </Button>
      </div>
    </form>
  );
};
