"use client";
// Start import
import { useState, useEffect } from "react";
import { Button } from "../../../../components/ui/button";
import { Label } from "../../../../components/ui/label";
import Select from "react-select";
import instance from "../../../../lib/axiosInstance";
import { useForm, Controller } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as yup from "yup";
import { Loader2 } from "lucide-react";
import { useCrudOperation } from "../../../context/CrudOperationContect";
import { toast } from "sonner";
// End import

// Schéma de validation YUP
const schema = yup.object().shape({
  name: yup.string().required("Le nom de la matière est requis"),
  level_id: yup
    .array()
    .min(1, "Sélectionnez au moins un niveau")
    .required("Le niveau est requis"),
});

// Export
export const Form = ({ closeModal, rowData }) => {
  // Set state for levelsOption
  const [levelOptions, setLevelOptions] = useState([]);

  // Utilisation du contexte pour accéder aux données et fonctions
  const { addOperation, updateOperation } = useCrudOperation();


  // Set useEffect hook
  useEffect(() => {
    const fetchLevelOptions = async () => {
      try {
        const response = await instance.get("settings/levels");
        const rawData = response.data.response.data;
        const formattedOptions = rawData.map((level) => ({
          value: level.id.toString(),
          label: level.name,
        }));
        setLevelOptions(formattedOptions);
      } catch (error) {
        console.error("Erreur chargement niveaux :", error);
        toast.error("Erreur", {
          description: "Impossible de charger les niveaux",
          position: "top-center",
        });
      }
    };
    fetchLevelOptions();
  }, []);

  // Format pre-selected data
  const defaultSelectedLevels = rowData?.levels.map((level) => ({
    value: level.id,
    label: level.name,
  }));

  // Define react-hook-form
  const {
    control,
    register,
    handleSubmit,
    formState: { errors, isSubmitting },
  } = useForm({
    resolver: yupResolver(schema),
    defaultValues: {
      name: rowData?.name,
      description: rowData?.description,
      level_id: defaultSelectedLevels,
    },
  });

  // Set method to add
  const onSubmit = async (data) => {
    try {
      // Formatage des données pour l'API
      const formattedData = {
        name: data.name,
        levels: data.level_id.map((level) => level.value),
      };

      // Set query
      const query = await instance.post("settings/subjects", formattedData);

      // Add to state
      addOperation(query.data.response);

      // Set response
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });

      // Ferme le modal après soumission réussie
      closeModal();
    } catch (error) {
      toast.error("Erreur", {
        // Correction de toast.success en toast.error
        description: error.response?.data?.message || "Une erreur est survenue",
        position: "top-center",
        style: {
          backgroundColor: "#E11409FF",
          color: "white",
        },
      });
    }
  };

  // Set method to update
  const onUpdate = async (data) => {
    try {
      // Formatage des données pour l'API
      const formattedData = {
        name: data.name,
        levels: data.level_id.map((level) => level.value),
      };
      // console.log(data);
      // console.log(rowData?.id);

      // Set query
      const query = await instance.put("settings/subjects/"+ rowData?.id, formattedData);

      // Add to state
      updateOperation(query.data.response);

      // Set response
      toast.success("Succès", {
        description: query.data.message,
        position: "top-center",
        style: {
          backgroundColor: "#4CAF50",
          color: "white",
        },
      });

      // Ferme le modal après soumission réussie
      closeModal();

    } catch (error) {
      errors.name.forEach((item) => {
        toast.error("Erreur", {
          description: item,
          position: "top-center",
          style: {
            backgroundColor: "#E11409FF",
            color: "white",
          },
        });
      });
    }
  };

  // Return
  return (
    <form
      onSubmit={rowData?.id ? handleSubmit(onUpdate) : handleSubmit(onSubmit)}
    >
      <div className="">
        <Label htmlFor="name" className="text-right mb-2">
          Matière
        </Label>
        <input
          id="name"
          {...register("name")}
          placeholder="Matière..."
          className="px-4 py-3.5 bg-white w-full text-sm border-2 border-gray-200 focus:border-[#fb923d] rounded-md outline-none text-black"
        />
        <span className="text-red-500 text-sm">{errors.name?.message}</span>
      </div>

      <div className="py-4">
        <Label htmlFor="level_id" className="text-right mb-2">
          Niveau
        </Label>
        <Controller
          name="level_id"
          control={control}
          render={({ field }) => (
            <Select
              options={levelOptions}
              isMulti
              value={field.value}
              onChange={(options) => field.onChange(options)}
              placeholder="Choisir un niveau"
              classNamePrefix="select"
            />
          )}
        />
        <span className="text-red-500 text-sm">{errors.level_id?.message}</span>
      </div>

      <div className="flex justify-end mt-4">
        <Button type="submit" disabled={isSubmitting}>
          {isSubmitting ? (
            <Loader2 className="animate-spin w-5 h-5 mr-2" />
          ) : (
            "Enregistrer"
          )}
        </Button>
      </div>
    </form>
  );
};
