"use client";
import * as React from "react";
import { createContext, useContext, useState, useEffect } from "react";
import instance from "../../lib/axiosInstance";

// Création du contexte
const CrudOperationContext = createContext();

// Hook personnalisé pour utiliser le contexte
export function useCrudOperation() {
  const context = useContext(CrudOperationContext);
  if (!context) {
    throw new Error(
      "useCrudOperation doit être utilisé à l'intérieur d'un CrudOperationProvider"
    );
  }
  return context;
}

// Provider du contexte
export function CrudOperationProvider({ children, endpoint }) {
  const [data, setData] = useState([]);
  const [loading, setLoading] = useState(true);
  const [refreshTrigger, setRefreshTrigger] = useState(0); // Nouvel état pour forcer le rafraîchissement

  // Fonction pour récupérer les données
  const fetchData = async () => {
    try {
      setLoading(true);
      // Appel à l'API avec Axios
      const response = await instance.get(endpoint);
      // Vérifiez la structure de la réponse et adaptez selon votre API
      const datas = response.data.response?.data ? response.data.response.data: response.data.response ;
      // console.log(datas);
      // Set datas to state
      setData(datas);
    } catch (err) {
      // console.error("Erreur lors de la récupération des données:", err);
    } finally {
      setLoading(false);
    }
  };

  // Fonction pour déclencher manuellement le rafraîchissement
  const handleRefresh = () => {
    setRefreshTrigger(prev => prev + 1); // Incrémente pour déclencher l'effet
  };

  // Fonction pour ajouter un niveau à la liste
  const addOperation = (newCrudOperation) => {
    setData((prevCrudOperation) => [...prevCrudOperation, newCrudOperation]);
  };

  // Fonction pour supprimer un element de la liste
  const deleteOperation = (idToDelete) => {
    setData((prevCrudOperation) =>
      prevCrudOperation.filter((row) => row.id !== idToDelete)
    );
  };

  const deleteOperationByResource = (idToDelete) => {
    setData((prevCrudOperation) =>
      prevCrudOperation.filter((row) => row.subject.id !== idToDelete)
    );
  };

  // Fonction pour mettre à jour un élément dans la liste
  const updateOperation = (updatedItem) => {
    setData((prevCrudOperation) =>
      prevCrudOperation.map((item) =>
        item.id === updatedItem.id ? updatedItem : item
      )
    );
  };

  // Effet pour charger les données au montage du composant
  useEffect(() => {
    fetchData();
  }, [endpoint, refreshTrigger]); // Déclenché quand endpoint ou refreshTrigger change

  // Valeurs et fonctions à partager via le contexte
  const value = {
    dataList: data,
    loading,
    addOperation,
    deleteOperation,
    deleteOperationByResource,
    updateOperation,
    // refreshOperation: fetchData,
    refreshOperation: handleRefresh,
  };

  return (
    <CrudOperationContext.Provider value={value}>
      {children}
    </CrudOperationContext.Provider>
  );
}
